<?php
require_once '../config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: /register.php');
    exit;
}

$full_name = trim($_POST['full_name'] ?? '');
$mobile_number = trim($_POST['mobile_number'] ?? '');
$email = trim($_POST['email'] ?? '');
$password = $_POST['password'] ?? '';
$confirm_password = $_POST['confirm_password'] ?? '';

// Validate inputs
if (empty($full_name) || empty($mobile_number) || empty($email) || empty($password) || empty($confirm_password)) {
    $_SESSION['error'] = 'Please fill in all fields';
    header('Location: /register.php');
    exit;
}

// Validate mobile number format
if (!preg_match('/^[0-9]{10,15}$/', $mobile_number)) {
    $_SESSION['error'] = 'Please enter a valid mobile number (10-15 digits)';
    header('Location: /register.php');
    exit;
}

// Validate email
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $_SESSION['error'] = 'Please enter a valid email address';
    header('Location: /register.php');
    exit;
}

// Validate password length
if (strlen($password) < 6) {
    $_SESSION['error'] = 'Password must be at least 6 characters long';
    header('Location: /register.php');
    exit;
}

// Check if passwords match
if ($password !== $confirm_password) {
    $_SESSION['error'] = 'Passwords do not match';
    header('Location: /register.php');
    exit;
}

try {
    // Check if mobile number already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE mobile_number = ?");
    $stmt->execute([$mobile_number]);
    if ($stmt->fetch()) {
        $_SESSION['error'] = 'This mobile number is already registered';
        header('Location: /register.php');
        exit;
    }

    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        $_SESSION['error'] = 'This email is already registered';
        header('Location: /register.php');
        exit;
    }

    // Hash password
    $hashed_password = password_hash($password, PASSWORD_DEFAULT);

    // Insert new user (let PostgreSQL generate the ID)
    $stmt = $pdo->prepare("
        INSERT INTO users (full_name, mobile_number, email, password, balance, created_at, updated_at) 
        VALUES (?, ?, ?, ?, 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
        RETURNING id
    ");
    $stmt->execute([$full_name, $mobile_number, $email, $hashed_password]);
    $newUser = $stmt->fetch();

    // Registration successful
    $_SESSION['success'] = 'Registration successful! Please login.';
    header('Location: /login.php');
    exit;

} catch (PDOException $e) {
    // Check for unique constraint violation
    if (strpos($e->getMessage(), 'unique') !== false || strpos($e->getMessage(), 'duplicate') !== false) {
        $_SESSION['error'] = 'Mobile number or email already registered';
    } else {
        $_SESSION['error'] = 'An error occurred. Please try again.';
    }
    header('Location: /register.php');
    exit;
}
?>
